# 通知設計書 6-Method Not Allowed通知

## 概要

本ドキュメントは、Horse Webフレームワークにおける Method Not Allowed通知機能の設計を記述する。リクエストされたHTTPメソッドがルートでサポートされていない場合にクライアントへHTTP 405エラーを返却する通知機能である。

### 本通知の処理概要

本通知は、HTTPリクエストのパスは存在するがリクエストされたHTTPメソッド（GET、POST、PUT等）がそのルートで定義されていない場合に、クライアントに対してHTTP 405 Method Not Allowedレスポンスを返却する機能を提供する。

**業務上の目的・背景**：RESTful APIにおいて、リソースに対して許可されていないHTTPメソッドが使用された場合、クライアントに適切なエラーレスポンスを返すことはHTTP仕様に準拠した基本要件である。この通知により、クライアントアプリケーションは使用すべき正しいHTTPメソッドを判断でき、APIの誤使用を早期に検出できる。404 Not Foundとの区別により、パスは正しいがメソッドが間違っていることを明確に伝達できる。

**通知の送信タイミング**：ルーティング処理において、リクエストパスに一致するルートは存在するが、リクエストされたHTTPメソッドに対応するコールバックが登録されていない場合に送信される。

**通知の受信者**：HTTPリクエストを送信したクライアント（Webブラウザ、モバイルアプリ、他のサービス等）。

**通知内容の概要**：レスポンスボディに「Method Not Allowed」というメッセージが含まれ、HTTPステータスコード405が返却される。

**期待されるアクション**：クライアントはメソッドエラーを検知し、正しいHTTPメソッドを使用してリクエストを再送する、またはユーザーに適切なエラーメッセージを表示する。

## 通知種別

HTTPレスポンス - HTTP 405 Method Not Allowed

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（HTTPレスポンス） |
| 優先度 | 即時 |
| リトライ | 無し（クライアント側で判断） |

### 送信先決定ロジック

HTTPリクエストを送信したクライアントに対してレスポンスとして返却される。送信先は自動的にリクエスト元に決定される。

## 通知テンプレート

### HTTPレスポンスの場合

| 項目 | 内容 |
|-----|------|
| HTTPステータスコード | 405 |
| ステータスメッセージ | Method Not Allowed |
| Content-Type | text/html (デフォルト) |
| 形式 | テキスト |

### 本文テンプレート

```
Method Not Allowed
```

### 添付ファイル

| ファイル名 | 形式 | 条件 | 説明 |
|----------|------|------|------|
| なし | - | - | HTTPレスポンスボディのみ |

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| なし | - | 固定メッセージのため変数なし | - |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| ルーティング | TNextCaller.Nextメソッド | パスにルート存在 かつ HTTPメソッドのコールバックなし かつ 他メソッドのコールバックあり | ルートは存在するがメソッドが許可されていない |

**送信条件の詳細**:
1. `FCallBack.TryGetValue(FHTTPType, LCallback)` が **False**を返す
2. `FCallBack.Count > 0` が **True**（他のメソッドのコールバックは存在）

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| パスが存在しない | パス自体が見つからない場合は404 Not Foundが返される |
| どのメソッドもコールバックなし | FCallBack.Count = 0 の場合は404 Not Foundが返される |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[HTTPリクエスト受信] --> B[ルーティング]
    B --> C{パスに一致するルート?}
    C -->|なし| D[404 Not Found]
    C -->|あり| E[TNextCaller.Next呼び出し]
    E --> F{リクエストメソッドのコールバック?}
    F -->|あり| G[コールバック実行]
    F -->|なし| H{他メソッドのコールバック?}
    H -->|あり| I[405エラーレスポンス送信]
    I --> J[Method Not Allowed]
    H -->|なし| D
```

## データベース参照・更新仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 備考 |
|-----------|------|------|
| なし | - | 本通知はデータベースを使用しない |

### 更新テーブル一覧

| テーブル名 | 操作 | 概要 |
|-----------|------|------|
| なし | - | 本通知はデータベースを更新しない |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| メソッド不一致 | GETルートにPOSTリクエスト等 | 405レスポンスを返却 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | サーバー側では行わない（クライアント側で判断） |
| リトライ間隔 | - |
| リトライ対象エラー | - |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 制限なし |
| 1日あたり上限 | 制限なし |

### 配信時間帯

HTTPリクエスト処理中に即座に送信される。時間帯の制限はない。

## セキュリティ考慮事項

- 固定メッセージ「Method Not Allowed」のみが返却されるため、内部情報の漏洩リスクは低い
- ルートの存在は確認されるため、パスの存在自体は攻撃者に推測される可能性がある
- RFC 7231によると、405レスポンスにはAllowヘッダーで許可されたメソッドを含めるべきだが、現在の実装では含まれていない

## 備考

- 404 Not Foundとの違いに注意：405はパスは存在するがメソッドが許可されていない、404はパス自体が存在しない
- THTTPStatus.MethodNotAllowedはHorse.Commons.pasで定義されている（値: 405）
- FFound^ := True が設定されるため、後続の404処理は行われない

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

HTTPメソッドとコールバック辞書の関係を理解することが重要。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Horse.Commons.pas | `src/Horse.Commons.pas` | THTTPStatus列挙型、特にMethodNotAllowed = 405 |
| 1-2 | Horse.Core.RouterTree.NextCaller.pas | `src/Horse.Core.RouterTree.NextCaller.pas` | FCallBack: TObjectDictionary<TMethodType, TList<THorseCallback>>（行33） |

**読解のコツ**: FCallBackはHTTPメソッドをキー、コールバックリストを値とする辞書。特定のメソッドに対するコールバックが存在するかをTryGetValueで確認する。

#### Step 2: エントリーポイントを理解する

メソッド判定とエラーレスポンスの流れを把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Horse.Core.RouterTree.NextCaller.pas | `src/Horse.Core.RouterTree.NextCaller.pas` | Nextメソッド（行80-131）のelse分岐（行115-125） |

**主要処理フロー**:
1. **行95**: `FCallBack.TryGetValue(FHTTPType, LCallback)` - 該当メソッドのコールバック取得試行
2. **行115-125**: TryGetValueがFalseの場合の処理
3. **行117-122**: else分岐 - メソッドが見つからない場合
4. **行118**: `if FCallBack.Count > 0 then` - 他のメソッドのコールバックが存在するか
5. **行119-121**: 存在する場合 - **ここが405エラー通知の送信箇所**
   - `FFound^ := True`
   - `FResponse.Send('Method Not Allowed').Status(THTTPStatus.MethodNotAllowed)`
6. **行123-124**: 存在しない場合 - 404 Not Found

#### Step 3: 404との分岐を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Horse.Core.RouterTree.NextCaller.pas | `src/Horse.Core.RouterTree.NextCaller.pas` | 行123-124の404処理との対比 |

**条件分岐の詳細**:
```pascal
if FCallBack.TryGetValue(FHTTPType, LCallback) then
  // コールバック実行
else
begin
  if FCallBack.Count > 0 then
  begin
    FFound^ := True;
    FResponse.Send('Method Not Allowed').Status(THTTPStatus.MethodNotAllowed);  // 405
  end
  else
    FResponse.Send('Not Found').Status(THTTPStatus.NotFound);  // 404
end;
```

### プログラム呼び出し階層図

```
THorseRouterTree.Execute (行139)
    │
    └─ THorseRouterTree.ExecuteInternal (行168)
           │
           └─ TNextCaller.Next (行80)
                  │
                  ├─ ミドルウェア実行 (行84-91)
                  │
                  └─ コールバック検索 (行92-131)
                         │
                         ├─ FCallBack.TryGetValue(FHTTPType, LCallback)
                         │      (行95)
                         │
                         └─ [TryGetValue = False の場合]
                                │
                                ├─ if FCallBack.Count > 0 then (行118)
                                │      │
                                │      ├─ FFound^ := True (行120)
                                │      │
                                │      └─ FResponse.Send('Method Not Allowed')
                                │             .Status(THTTPStatus.MethodNotAllowed)
                                │             (行121)
                                │
                                └─ else (行123)
                                       │
                                       └─ FResponse.Send('Not Found')
                                              .Status(THTTPStatus.NotFound)
                                              (行124)
```

### データフロー図

```
[入力]                      [処理]                           [出力]

HTTPリクエスト     ───▶ ルーティング                   ───▶ HTTPレスポンス
POST /users               │                                  Status: 405
                          ▼                                  Body: "Method Not
                    FCallBack辞書検索                         Allowed"
                    TryGetValue(mtPost, ...)
                          │
                          ▼
                    [False - POSTコールバックなし]
                          │
                          ▼
                    FCallBack.Count確認
                          │
                          ▼
                    [> 0 - GETコールバックあり]
                          │
                          ▼
                    FFound^ := True
                          │
                          ▼
                    FResponse.Send('Method Not Allowed')
                          │
                          ▼
                    .Status(THTTPStatus.MethodNotAllowed)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Horse.Core.RouterTree.NextCaller.pas | `src/Horse.Core.RouterTree.NextCaller.pas` | ソース | ルーティング判定と405エラー通知の送信元 |
| Horse.Commons.pas | `src/Horse.Commons.pas` | ソース | THTTPStatus列挙型、MethodNotAllowed = 405 |
| Horse.Core.RouterTree.pas | `src/Horse.Core.RouterTree.pas` | ソース | ルーティングツリー管理、Execute呼び出し元 |
| Horse.Response.pas | `src/Horse.Response.pas` | ソース | THorseResponse.Send/Statusメソッド |
